### 10/12/2018, Satbyeol Shin, University of Florida ###
### mlm: Maximum Likelihood Method
#install.packages("maxLik")
library(maxLik)
#install.packages("lmomco")
library(lmomco)
#install.packages("ismev")
library(ismev)
### function: using mlm, parameter estimation for Normal Distribution (nor)
# Rao and Hamed,2000.Flood frequency analysis. p.87 (eq.5.1.16)
mlm_nor <- function(x) {
  mlm_norf <- function(tet,x) {
    mu <- tet[1] 
    sigma <- tet[2]
    N <- length(x)
    flik <- (-N)*log(sigma)-N*log((2*pi)^0.5)-(1/(2*sigma^2))*sum((x-mu)^2)
    return (flik)
  }
  a <- maxLik(mlm_norf,x=x,start=c(1,1)) #success
  mean <- a$estimate[1]
  sd <- a$estimate[2]
  f <- cbind(mean,sd)
  return(f)
}
### function end ###
### function: using mlm, parameter estimation for Two-Parameter Lognormal Distribution (ln2)
# Rao and Hamed,2000.Flood frequency analysis. p.98 (eq.5.2.10)
mlm_ln2 <- function(x) {
  mlm_ln2f <- function(tet,x) {
    mu <- tet[1]
    sigma <- tet[2]
    N <- length(x)
    flik <- sum(log(1/x))-(N/2)*log(sigma^2)-(1/(2*sigma^2))*sum((log(x)-mu)^2)
    return(flik)
  }
  a <- maxLik(mlm_ln2f,x=x,start=c(1,1))
  mean <- a$estimate[1]
  sd <- a$estimate[2]
  f <- cbind(mean,sd)
  return(f)
}
### function end ###
### function: using mlm, parameter estimation for Three-Parameter Lognormal Distribution (ln3)
# Rao and Hamed,2000.Flood frequency analysis. p.110 (eq.5.3.14)
mlm_ln3 <- function(x) {
  mlm_ln3f <- function(tet,x) { #tet[1]=location, tet[2]=scale, tet[3]=shape
    a <- tet[1]
    mu <- tet[2]
    sigma <- tet[3]
    N <- length(x)
    flik <- -sum(log(x-a))-(N/2)*log(2*pi)-(N/2)*(log(sigma^2))-(1/(2*sigma^2))*sum((log(x-a)-mu)^2)
    return(flik)
  }
  a <- maxLik(mlm_ln3f,x=x,start=c(0.01,0.01,0.01))
  thres <- a$estimate[1]
  scale <- a$estimate[2]
  shape <- a$estimate[3]
  f <- cbind(shape,scale,thres)
  return(f)
}
### function end ###
### function: using mlm, parameter estimation for Exponential Distribution (exp)
mlm_exp <- function(x) {
  a <- mle2par(x,type="exp") # using packages("lmomco")
  if (is.null(a)==TRUE) {
    f <- 0
  } else {
    shift <- a$para[1]
    rate <- a$para[2]
    f <- cbind(shift,rate)
  }
  return(f)
}
### function end ###
### function: using mlm, parameter estimation for Two-Parameter Gamma Distribution (gam2)
# Rao and Hamed,2000.Flood frequency analysis. p.143 (eq.6.2.20)
mlm_gam2 <- function(x) {
  mlm_gam2f <- function(tet,x) { #tet[1]=scale, tet[2]=shape
    a <- tet[1]
    b <- tet[2]
    N <- length(x)
    flik <- -N*b*log(a)-N*log(lgamma(b))+(b-1)*sum(log(x))-(1/a)*sum(x)
    geo_mean <- function(data) {
      log_data <- log(data)
      gm <- exp(mean(log_data[is.finite(log_data)]))
      return(gm)
    }
    U <- log(mean(x))-log(geo_mean(x))
    if((U>=0) && (U<=0.5772)) {
      b <- 1/U*(0.5000876+0.1648852*U-0.054427*U^2)
    } else if ((U>=0.5772) && (U<=17.0)) {
      b <- (8.898919+9.059950*U+0.9775373*U^2)/(U*(17.7928+11.968477*U+U^2))
    }
    a <- mean(x)/b
    f <- cbind(a,b)
    return(f)
  }
  a <- mlm_gam2f(tet=c(100,100),x=x)
  scale <- a[1]
  shape <- a[2]
  f <- cbind(shape,scale)
  return(f)
}
### function end ###
### function: using mlm, parameter estimation for Three-Parameter Gamma Distribution (gam3) = Pearson type III
mlm_gam3 <- function(x) {
  m <- mean(x)
  sd <- sd(x)
  f <- mle2par(x,type="pe3") # using packages("lmomco")
  if (is.null(f)==TRUE) {
    res <- 0
  } else {
    gamma <- f$para[3]
    alpha <- 4/(gamma^2)
    beta <- 1/2*sd*abs(gamma)
    e <- m-2*sd/gamma
    res <- cbind(alpha, beta, e) # alpha: shape, beta: scale, e: thres
  }
  return(res)
}
### function end ###
### function: using mlm, parameter estimation for Log-Pearson type III distribution (lp3)
mlm_lp3 <- function(x) {
  x <- log(x)
  m <- mean(x)
  sd <- sd(x)
  f <- mle2par(x,type="pe3") # using packages("lmomco")
  if (is.null(f)==TRUE) {
    res <- 0
  } else {
    gamma <- f$para[3]
    alpha <- 4/(gamma^2) 
    beta <- 1/2*sd*abs(gamma)
    e <- m-2*sd/gamma
    res <- cbind(alpha, beta, e) # alpha: shape, beta: scale, e: thres
  }
  return(res)
}
### function end ###
### function: using mlm, parameter estimation for Generalized Extreme Value Distribution (gev)
mlm_gev <- function(x) {
  a <- gev.fit(x,show=F) # using packages("lsmev")
  location <- a$mle[1]
  scale <- a$mle[2]
  shape <- a$mle[3]
  f <- cbind(shape,scale,location)
  return(f)
}
### function end ###
### function: using mlm, parameter estimation for Gumbel Distribution (gum)
# Mahdi and Cenac,2005.Estimating parameters of Gumbel distribution using the methods of moments, probability weighted moments and maximum likelihood
mlm_gum <- function(x) {
  mlm_gumf <- function(tet,x) { 
    a <- tet[1]
    e <- tet[2]
    N <- length(x)
    flik <- -sum((x-e)/a)-N*log(a)-sum(exp(-(x-e)/a))
    return(flik)
  }
  a <- maxLik(mlm_gumf,x=x,start=c(1,1))
  scale <- a$estimate[1]
  location <- a$estimate[2]
  f <- cbind(scale,location)
  return(f)
}
### function end ###
### function: using mlm, parameter estimation for Weibull2 Distrubition (wbu2)
mlm_wbu2 <- function(x) {
  mlm_wbu2f <- function(tet,x) {
    g <- tet[1]
    r <- tet[2]
    flik <- sum(dweibull(x,shape=g, scale=r, log=TRUE))
    return(flik)
  }
  a <- maxLik(mlm_wbu2f,x=x,start=c(1,1))
  shape <- a$estimate[1]
  scale <- a$estimate[2]
  f <- cbind(shape,scale)
  return(f)    
}
### function end ###
### function: using mlm, parameter estimation for Weibull3 Distrubition (wbu3)
mlm_wbu3 <- function(x) {
  a <- mle2par(x,type="wei") # using packages("lmomco")
  if (is.null(a)==TRUE) {
    f <- c(0,0,0)
  } else {
    thres <- -a$para[1]
    scale <- a$para[2]
    shape <- a$para[3]
    f <- cbind(shape,scale,thres)
  }
  return(f)
}
### function end ###