### 10/12/2018, Satbyeol Shin, University of Florida ###
### mom: Method of Moments
#install.packages("e1071")
library(e1071) # for calculating skewness
#install.packages("nleqslv")
library(nleqslv)
### function: using mom, parameter estimation for Normal Distribution (nor)
mom_nor <- function(x) {
  m <- mean(x)
  sd <- sd(x)
  f <- cbind(m,sd) # m: mean, sd: sd
  return (f)
}
### function end ###
### function: using mom, parameter estimation for Two-Parameter Lognormal Distribution (ln2)
mom_ln2 <- function(x) {
  x1 <- log(x)
  m <- mean(x1)
  sd <- sd(x1)
  f <- cbind(m,sd) # m: mean, sd: sd
  return(f)
}
### function end ###
### function: using mom, parameter estimation for Three-Parameter Lognormal Distribution (ln3)
# Kite, 1988.Frequency and risk analyses in hydrology. p.73~74 (eq.7-12~7-17)
mom_ln3 <- function(x) {
  m <- mean(x)
  sd <- sd(x)
  m2 <- (exp(sd^2)-1)*(exp(2*m+sd^2))
  sk <- skewness(x)
  w <- (-sk+((sk^2)+4)^0.5)/2
  z2 <- (1-w^(2/3))/(w^(1/3))
  a <- m-sd/z2
  sigma <- (log(z2^2+1))^0.5
  mu <- log(sd/z2)+(-1/2)*log((z2^2)+1)
  f <- cbind(sigma,mu,a) # sigma: shape, mu: scale, a: thres
  return(f)
}
### function end ###
### function: using mom, parameter estimation for Exponential Distribution (exp)
# Rao and Hamed,2000.Flood frequency analysis. p.130 (eq.6.1.18~6.1.19)
mom_exp <- function(x) {
 a <- sd(x)
 e <- moment(x,order=1) - a
 f <- cbind(a,e) # a: shift, e: rate
 return(f)
}
### function end ###
### function: using mom, parameter estimation for Two-Parameter Gamma Distribution (gam2)
# Rao and Hamed,2000.Flood frequency analysis. p.142 (eq.6.2.17~6.2.18)
mom_gam2 <- function(x) {
  ex1 <- mean(x)
  ex2 <- mean(x^2)
  shape <- -ex1^2/(ex1^2-ex2)
  scale <- (-ex1^2+ex2)/ex1
  f <- cbind(shape,scale)
  return(f)
}
### function end ###
### function: using mom, parameter estimation for Three-Parameter Gamma Distribution (gam3) = Pearson type III
# Rao and Hamed,2000.Flood frequency analysis. p.157 (eq.6.3.15~6.3.17)
# Kite, 1988. Frequency and risk analyses in hydrology p.117, eq. 9-2
mom_gam3 <- function(x) {
  sk <- skewness(x)
  n <- length(x)
  sk <- sk*(n*(n-1))^0.5/(n-2)*(1+8.5/n)
  beta <- (2/sk)^2
  sd <- sd(x)
  mu <- mean(x)
  alpha <- sd/(beta)^0.5
  gamma <- mu-sd*(beta)^0.5
  f <- cbind(beta, alpha, gamma) # beta: shape, alpha: scale, gamma: thres
  return(f)
}
### function end ###
### function: using mom, parameter estimation for Log-Pearson type III distribution (lp3)
# Rao and Hamed,2000.Flood frequency analysis. p.176 (eq.6.4.12~6.4.19)
mom_lp3 <- function(x) {
  logm1 <- log(moment(x,order=1))
  logm2 <- log(moment(x,order=2))
  logm3 <- log(moment(x,order=3))
  B <- (logm3-3*logm1)/(logm2-2*logm1)
  C <- 1/(B-3)
  if (B>3.5 && B<6) {
    A <- -0.23019+1.65262*C+0.20911*C^2-0.04557*C^3
  } else if (B>3 && B<=3.5) {
    A <- -0.47157+1.99955*C
  } else A <- NULL
  if (is.null(A)==TRUE) {
    f <- 0
  } else {
    alpha <- 1/(A+3)
    beta <- (logm2-2*logm1)/(log((1-alpha)^2)-log(1-2*alpha))
    gamma <- logm1+beta*log(1-alpha)
    f <- cbind(beta, alpha, gamma) # beta: shape, alpha: scale, gamma: thres
  }
  return(f)
}
### function end ###
### function: using mom, parameter estimation for Generalized Extreme Value Distribution (gev)
# Rao and Hamed,2000.Flood frequency analysis. p.209~212 (eq.7.1.12~7.1.21)
mom_gev <- function(x) {
  m1 <- mean(x)
  m2 <- sd(x)^2
  cs <- skewness(x)
  if (cs>1.14 && cs<10) {
    k <- 0.2858211-0.357983*cs+0.116659*cs^2-0.022725*cs^3+0.002604*cs^4-0.000161*cs^5+0.000004*cs^6
    if (k<0) {
      k <- k
    } else {
      k <- NULL
    }
  } else if (cs>-2 && cs<1.14) {
    k <- 0.277648-0.322016*cs+0.060278*cs^2+0.016759*cs^3-0.005873*cs^4-0.00244*cs^5-0.00005*cs^6
    if (k>0) {
      k <- k
    } else {
      k <- NULL
    }
  } else if (cs>-10 && cs<0) {
    k <- -0.50405-0.00861*cs+0.015497*cs^2+0.005613*cs^3+0.00087*cs^4+0.000065*cs^5
    if (k<0) {
      k <- k
    } else {
      k <- NULL
    }
  }
  g1 <- gamma(1+2*k)
  g2 <- gamma(1+k)
  alpha <- ((m2*k^2)/(gamma(1+2*k)-(gamma(1+k))^2))^0.5
  u <- m1-(alpha/k)*(1-gamma(1+k))
  f <- cbind(k,alpha,u) # k: shape, alpha: scale, u: location
  return(f)
}
### function end ###
### function: using mom, parameter estimation for Gumbel Distribution (gum)
# Rao and Hamed,2000.Flood frequency analysis. p.232 (eq.7.2.5~7.2.6)
mom_gum <- function(x) {
  m1 <- mean(x)
  m2 <- sd(x)^2
  alpha <- 0.7796*m2^0.5
  beta <- m1-0.45005*m2^0.5
  f <- cbind(alpha,beta) # alpha: scale, beta: location
  return(f)
}
### function end ###
### function: using mom, parameter estimation for Weibull2 Distrubition (wbu2)
# Nielsen, 2011, Parameter Estimation for the Two-Parameter Weibull Distribution. p.8-10
mom_wbu2 <- function(x) {
  m1 <- mean(x)
  m2 <- moment(x,order=2)
  gam <- function(r) {
    g <- gamma(1+2/r)/(gamma(1+1/r))^2-m2/m1^2
    return(g)
  }
  gamma <- nleqslv(c(0.1),fn=gam)$x

  beta <- m1/gamma(1+1/gamma)
  f <- cbind(gamma,beta) # gamma: shape, beta: scale
  return(f)
}
### function end ###